/*
  client() is called from main() and implements the master process of
  r2sync. It forks two servers (unless instructed to use existing
  servers). It then communicates with them to get the list of changed
  files on both. Next it asks the user (unless r2sync is running in
  batch mode) what to do with them. And then it sends commands to the
  servers to synchronize or delete the files, as appropriate.

  Synchronizing a file consists of three steps for each file: (1) ask
  the server with the outdated file to compute a signature of the file
  (i.e., a series of checksums for each block of the file). (2) ask
  the server with the up-to-date file to compute a delta (i.e., a
  series of patches) given that signature and the up-to-date-file.
  (3) ask the server with the outdated file to apply those
  patches. For details, see synchronize() in "c-synchronize.c".
*/

#include "stdincls.h"
#include "types.e"
#include "connectsock.e"
#include "getline.e"
#include "print.e"
#include "s-serve.e"
#include "c-profile.e"
#include "c-progress.e"
#include "c-statistics.e"
#include "c-setroots.e"
#include "c-setproto.e"
#include "c-checkroots.e"
#include "c-getfiles.e"
#include "c-getactions.e"
#include "c-synchronize.e"
#include "c-delfiles.e"
#include "c-getproto.e"


/* fork_server -- fork a daemon on host and create pipes to and from it */
static pid_t fork_server(char *user, char *host, FILE **to_server,
			 FILE **from_server, bool compression)
{
# define MAXARGS 12
  int to[2], from[2], i;
  char *s, errcode[16], *args[MAXARGS];
  pid_t pid;

  assert(!user || host);	/* user never defined without host */
  errno = 0;

  if (pipe(to) == -1 || pipe(from) == -1) {warn(NULL); return -1;}

  /* Fork a process for the server */
  if ((pid = fork()) == -1) {warn(NULL); return -1;}

  if (pid == 0) {		/* Child */
    close(to[1]);
    close(from[0]);
    if (!host) {		/* Local */
      if (serve(to[0],from[1],NULL) || signaled) exit(0); else exit(EX_PROTOCOL);
    } else {			/* Remote */
      /* Redirect stdin & stdout to the two pipes */
      if (dup2(to[0], 0) == -1) err(EX_OSERR, NULL);
      close(to[0]);
      if (dup2(from[1], 1) == -1) err(EX_OSERR, NULL);
      close(from[1]);
      /* Start r2sync on the remote host */
      i = 0;
      args[i++] = "ssh";
      if (user) {args[i++] = "-l"; args[i++] = user;}
      args[i++] = "-x";
      args[i++] = "-a";
      args[i++] = "-k";
      if (compression) args[i++] = "-C";
      args[i++] = host;
      args[i++] = "--";
      args[i++] = "r2sync";
      args[i++] = "-d";
      args[i++] = NULL;
      assert(i <= MAXARGS);
      execvp("ssh", args);
      /* If we're here, an error occurred */
      s = strerror(errno);
      (void) sprintf(errcode, "? %03d ", 500 + errno);
      (void) write(1, errcode, strlen(errcode));
      (void) write(1, s, strlen(s));
      exit(EX_UNAVAILABLE);
    }
  }

  /* If we're here, we're in the parent. */
  close(to[0]);
  close(from[1]);

  *to_server = fdopen(to[1], "w");
  *from_server = fdopen(from[0], "r");

  return pid;
}


/* start_one_server -- start or connect to a server */
static bool start_one_server(char **root, pid_t *child, FILE **to,
			     FILE **from, char id[MACHINE_ID_LEN+1],
			     Profile profile)
{
  char *host = NULL, *user = NULL, *port;
  bool case_sensitive;
  int server_proto;
  size_t i, j;

  if (strncasecmp(*root, "r2sync://", 9) == 0) {

    i = 9 + strcspn(*root + 9, ":/");
    j = i + strcspn(*root + i, "/");
    host = strndup(*root + 9, i - 9);
    if ((*root)[i] == ':')
      port = strndup((*root) + i + 1, j - i - 1);
    else
      port = strdup("874");
    *to = *from = fconnectTCP(host, port);
    free(host);
    free(port);
    if (!*to) {warn(NULL); return false;}

  } else {

    /* root can be "user@host:path", "host:path", or "path" */
    i = strcspn(*root, "@:/");
    j = i + strcspn(*root + i, ":/");
    if ((*root)[j] != ':') {
      /* No host, so must be a local path. Leave host and user undefined. */
    } else if ((*root)[i] == '@') { /* root starts with "user@host:" */
      user = strndup(*root, i);
      host = strndup(*root + i + 1, j - i - 1);
    } else {			/* root starts with "host:" */
      host = strndup(*root, j);
    }
    *child = fork_server(user, host, to, from, profile->compression);
    free(user);
    free(host);
    if (*child == -1 || to == NULL || from == NULL) return false;
  }

  /* Read and write a line at a time */
  setlinebuf(*to);
  setlinebuf(*from);

  /* Get the "ready" line from the server */
  if (!get_protocol(*from, id, &server_proto, &case_sensitive)) return false;

  assert(profile->protocol > 0);

  /* Use the lowest of our and their protocol versions */
  if (server_proto < profile->protocol) profile->protocol = server_proto;

  /* Use case-insensitive comparisons if this server requires it */
  if (!case_sensitive) profile->nocase = true;

  debug("Server started\n");

  return true;
}


/* start_two_servers -- start the two servers, return the pipes and pids */
static bool start_two_servers(Profile profile)
{
  return
    start_one_server(&profile->root1, &profile->child1, &profile->to1,
		     &profile->from1, profile->id1, profile) &&
    start_one_server(&profile->root2, &profile->child2, &profile->to2,
		     &profile->from2, profile->id2, profile);
}


/* stop_server -- stop a server, close the pipes */
static void stop_server(pid_t *child, FILE **to, FILE **from, const char *target)
{
  int stat_loc;

  if (*child > 0) {
    if (*to) {print1(*to, "quit\n"); (void) fclose(*to); *to = NULL;}
    if (*from) {(void) fclose(*from); *from = NULL;}
    waitpid(*child, &stat_loc, 0);
    *child = 0;

    if (WIFSIGNALED(stat_loc))
      warnx(_("%1$s terminated by signal %2$d"), target, WTERMSIG(stat_loc));
    else if (/*WIFEXITED(stat_loc) &&*/ WEXITSTATUS(stat_loc) != 0)
      warnx(_("\"%1$s\" exited with status %2$d"), target, WEXITSTATUS(stat_loc));
  }
}


/* stop_servers -- stop the servers, close the pipes */
static void stop_servers(Profile profile)
{
  stop_server(&profile->child1, &profile->to1, &profile->from1, profile->root1);
  stop_server(&profile->child2, &profile->to2, &profile->from2, profile->root2);
}


/* client -- start two servers and synchronize root1 and root2 */
EXPORT bool client(Profile profile)
{
  int nfiles1 = 0, nfiles2 = 0;
  fileinfo *files1 = NULL, *files2 = NULL;
  fileinfo *send_to1 = NULL, *send_to2 = NULL;
  char **delete = NULL;
  int nsend1 = 0, nsend2 = 0, ndelete = 0;
  bool result;

  assert(profile->root1 && profile->root2);

  result =
    start_two_servers(profile) &&
    set_protocol(profile) &&
    set_roots(profile) &&
    check_root_compatibility(profile) &&
    get_file_lists(profile, &files1, &nfiles1, &files2, &nfiles2) &&
    determine_file_actions(profile, files1, nfiles1, files2, nfiles2,
			   &send_to1, &nsend1, &send_to2, &nsend2,
			   &delete, &ndelete) &&
    init_progress_meter(profile, send_to1, nsend1, send_to2, nsend2, ndelete);
  if (result) {
    result = synchronize(profile, send_to1, nsend1, send_to2, nsend2);
    if (result || profile->keep_going)
      result = delete_files(profile, delete, ndelete) && result;
    if (profile->statistics)
      print_statistics(profile, files1, nfiles1, files2, nfiles2);
  }

  /* Finalize */
  debug("Finalizing...\n");

  stop_servers(profile);

  debug("Freeing memory...\n");

  while (nfiles1) free(files1[--nfiles1].path);
  while (nfiles2) free(files2[--nfiles2].path);
  free(files1);
  free(files2);
  free(send_to1);
  free(send_to2);
  free(delete);

  if (!result) warnx(_("Interrupted"));
  else if (signaled) warnx(_("Interrupted by signal %d"), signaled);
  return result;
}
