/*
  set_roots() is called by client(). Its task is to tell the two
  servers the names of the directories or files to synchronize and to
  ask them if they are indeed directories or files or something else.

  Implementation in protocol 1:

  A target may have the form "r2sync://<connection><path>", where
  <path> is an absolute path on <host>; or "<user>@<host>:<path>",
  where <path> may be relative or absolute; or just "<path>".
  (start_one_server() already made sure that a <host> is prefixed with
  a <user>.)

  Each server receives "local <path>" and replies either with an error
  or with "<type> <realpath>", where <realpath> is the absolute path
  of <path> with all symlinks resolved; and <type> is "file",
  "directory" or "other".

  Each server then receives the target of the other server, either
  "remote r2sync://<connection><realpath>" or "remote
  <user>@<host>:<realpath>". The server replies with "OK" or an error.

  If the command line had the option -r (--reset), each server then
  receives a "reset" command, to tell it to discard any old log for
  these targets.
*/

#include "stdincls.h"
#include "types.e"
#include "getline.e"
#include "print.e"
#include "c-profile.e"


/* set_local -- tell a server what target to sync, return full name and type */
static bool set_local(FILE *to, FILE *from, char *root, filetype *type,
		      char **realpath)
{
  size_t i;
  char *line = NULL;
  bool ok;

  /* Tell the server the path to sync. */
  if (strncasecmp(root, "r2sync://", 9) == 0)
    ok = print(to, "local %s\n", root + 9 + strcspn(root + 9, "/")) > 0;
  else {
    i = strcspn(root, ":/");
    ok = print(to, "local %s\n", root[i] == ':' ? root + i + 1 : root) > 0;
  }

  /* Get the response: "<type> <realpath>" */
  if (ok) ok = (line = getline_chomp(from)) && line[0] != '?';

  if (ok) {
    i = strcspn(line, " ");
    if (strncmp(line, "directory", i) == 0) *type = IS_DIR;
    else if (strncmp(line, "file", i) == 0) *type = IS_FILE;
    else *type = IS_OTHER;

    if (!(*realpath = realloc(*realpath, 1 + strlen(line + i + 1)))) ok = false;
    else strcpy(*realpath, line + i + 1);
  }

  /* Inform user if an error occurred */
  if (!ok) {
    /* l10n: The first %s is the name of a target, the second
       is a raw (English) error message from the protocol. */
    if (line && line[0] == '?') warnx(_("Error on '%1$s': %2$s"), root, line + 2);
    else if (errno) warn(_("Error on '%s'"), root);
    else warnx(_("Error on '%s'"), root);
  }

  return ok;
}


/* set_remote -- tell a server what the remote target is */
static bool set_remote(FILE *to, FILE *from, const char *remote_id,
		       const char *remote_path)
{
  char *line = NULL;
  bool ok;

  /* Tell the server the machine ID and path of the other server */
  ok = print(to, "remote %s %s\n", remote_id, remote_path);

  /* Response should be "OK" or an error */
  if (ok) ok = (line = getline1(from)) && line[0] != '?';

  /* Inform user if an error occurred */
  if (!ok) {
    if (line && line[0] == '?') warnx(_("Error: %s"), line + 2);
    else if (errno) warn(_("Error"));
    else warnx(_("Error"));
  }

  return ok;
}


/* reset -- tell a server to discard its log */
static bool reset(FILE *to, FILE *from)
{
  char *line = NULL;
  bool ok;

  ok = print1(to, "reset\n") > 0 &&
    (line = getline1(from)) && line[0] != '?';

  /* Inform user if an error occurred */
  if (!ok) {
    if (line && line[0] == '?') warnx(_("Error with 'reset': %s"), line + 2);
    else if (errno) warn(_("Error with 'reset'"));
    else warnx(_("Error with 'reset'"));
  }

  return ok;
}


/* set_roots -- tell the servers the roots/paths to synchronize */
EXPORT bool set_roots(Profile p)
{
  /* This also stores the machine IDs and real paths in the profile,
     based on the responses to the "local" commands. */
  return
    set_local(p->to1, p->from1, p->root1, &p->type1, &p->path1) &&
    set_local(p->to2, p->from2, p->root2, &p->type2, &p->path2) &&
    set_remote(p->to1, p->from1, p->id2, p->path2) &&
    set_remote(p->to2, p->from2, p->id1, p->path1) &&
    (!p->reset_logs || reset(p->to1, p->from1)) &&
    (!p->reset_logs || reset(p->to2, p->from2));
}
