/*
  say_hello() is called by serve1() right after a connection is
  established to tell the client what the server's version and
  characteristics are.

  Implementation in protocol 1:

  say_hello() sends on line

      ready ID N
  or
      ready ID N nocase

  ID is a 32-character, lowercase hexadecimal string that uniquely
  identifies the server's computer. N is the protocol version in
  decimal (currently 1). If the local file system is case-insensitive,
  say_hello() adds "nocase".
*/

#include "stdincls.h"
#include "types.e"
#include "print.e"
#include "homedir.e"
#include "errcodes.e"
#include "export.h"


/* system_is_case_sensitive -- check if the file system is case-sensitive */
static bool system_is_case_sensitive(void)
{
  char lowercase[] = "/tmp/r2sync-XXXXXX";
  char uppercase[] = "/tmp/R2SYNC-XXXXXX";
  bool case_sensitive = true;
  int fd;

  /* Try to create a lowercase file and re-open it as uppercase. On a
     case-sensitive file system, that should fail. (This test could
     fail for other reason, but chances are low.)
  */
  if ((fd = mkstemp(lowercase)) != -1) {
    close(fd);
    strcpy(uppercase + 12, lowercase + 12);
    fd = open(uppercase, O_RDONLY, 0666);
    case_sensitive = (fd == -1);
  }
  if (fd != -1) {close(fd); unlink(lowercase);}	/* Clean up */

  return case_sensitive;
}


/* read_id -- read an ID from file and check it */
static char* read_id(const char *dir, const char *file)
{
  static char id[MACHINE_ID_LEN + 2];
  ssize_t n;
  size_t i;
  char *s;
  int fd;

  if (!(s = malloc(strlen(dir) + strlen(file) + 2))) return NULL;
  sprintf(s, "%s/%s", dir, file);
  fd = open(s, O_RDONLY);
  free(s);
  if (fd == -1) return NULL;
  n = read(fd, id, sizeof(id) - 1);
  close(fd);
  if (n == -1) return NULL;
  if (n > 0 && id[n-1] == '\n') n--;
  if (n > 0 && id[n-1] == '\r') n--;
  id[n] = '\0';
  debug("read id (%d bytes): \"%s\"\n", n, id);
  i = strspn(id, "0123456789abcdef"); /* Check that it is lowercase hex */
  if (i != MACHINE_ID_LEN) return NULL;
  return id;
}


/* write_id -- generate a random, 32-char hex string and write it to file */
static char* write_id(char *dir, char *file)
{
  static char id[MACHINE_ID_LEN + 2];
  ssize_t n;
  size_t i;
  char *s;
  int fd;
  long r;

  if (!create_directories(dir)) return NULL;
  if (!(s = malloc(strlen(dir) + strlen(file) + 2))) return NULL;
  sprintf(s, "%s/%s", dir, file);
  fd = open(s, O_WRONLY|O_CREAT, 0644);
  free(s);
  if (fd == -1) return NULL;
  srandom(time(NULL));
  for (i = 0; i < MACHINE_ID_LEN; i++) {
    r = random() % 16;
    id[i] = r < 10 ? '0' + r : 'a' + r - 10;
  }
  n = write(fd, id, i);
  close(fd);
  if (n < i) return NULL;
  id[i] = '\0';
  return id;
}


/* get_unique_system_id -- return a unique identifier for this computer */
static char* get_unique_system_id(void)
{
  char *home, *s;
  size_t n;
  char *id;

  /* If there is a file /etc/machine-id containing a 32-character hex
     number (possibly followed by a newline), use that number.
     Otherwise, if there is a file ~/.r2sync/machine-id containing a
     hex number, use that number. And if there is neither, create a
     new ~/.r2sync/machine-id.
  */
  if ((id = read_id("/etc", "machine-id"))) return id;

  if (!(home = home_dir())) return NULL;
  n = strlen(home) + sizeof(R2SYNCDIR) + 2;
  if (!(s = malloc(n))) {free(home); return NULL;}
  sprintf(s, "%s/%s", home, R2SYNCDIR);
  free(home);

  id = read_id(s, "machine-id");
  if (!id) id = write_id(s, "machine-id");
  free(s);
  return id;
}


/* say_hello -- print server's version and capabilities */
EXPORT void say_hello(FILE *out, bool *case_sensitive)
{
  char *system_id;

  *case_sensitive = system_is_case_sensitive();
  system_id = get_unique_system_id();

  if (!system_id)
    print(out, "? %03d Cannot get system ID\n", EC_NOSYSTEMID);
  else				/* Print the hello string */
    print(out, "ready %s %d%s\n", system_id, R2SYNCPROTO,
	  *case_sensitive ? "" : " nocase");
}
