/*
 * ask() is called by determine_file_actions() to ask the user what to
 * do with a pair of files: synchronize a to b, b to a, or skip.
 *
 * It returns a one-character response. See getactions.c for their
 * descriptions.
 *
 * In addition, ask() handles the '?' action itself, by showing a help
 * text.
 */

#include "stdincls.h"
#include "types.e"


/* get_keypress -- get one key press in terminal raw mode */
EXPORT int get_keypress(void)
{
  struct termios terminal_settings, raw_settings;
  int c;

  (void) tcgetattr(0, &terminal_settings);
  raw_settings = terminal_settings;
  cfmakeraw(&raw_settings);	/* Get settings for a raw terminal... */
  raw_settings.c_lflag |= ISIG;	/* ... but still allow Ctrl-C ("INTR") */
  if (tcsetattr(0, TCSAFLUSH, &raw_settings) == -1) warn(NULL);
  c = getchar();
  (void) tcsetattr(0, TCSANOW, &terminal_settings);
  return c;
}


/* ask -- ask the user what to do with two updated files */
EXPORT int ask(fileinfo *file1, fileinfo *file2, char default_action)
{
  const char *s1, *s2;
  char *oper, *path, *s3;
  int answer = ' ';

  assert(file1 || file2);

  /* Loop until we got a valid answer (or an interrupt) */
  while (!strchr("<>/xpgq", answer) && !signaled) {

    /* l10n: The following ten strings must all be the same
       width when printed (but not necessarily the same width as the
       English string). The first five are right-aligned, the second
       five are the same, but left-aligned. */
    if (!file1)                    s1 =   "         ";
    else if (file1->status == '=') s1 = _("unchanged");
    else if (file1->status == 'n') s1 = _("      new");
    else if (file1->status == 'd') s1 = _("  deleted");
    else if (file1->status == 'm') s1 = _("     mode");
    else                           s1 = _("  changed");

    if (default_action == '<') oper = "<--";
    else if (default_action == '>') oper = "-->";
    else oper = "<->";

    if (!file2)                    s2 =   "         ";
    else if (file2->status == '=') s2 = _("unchanged");
    else if (file2->status == 'n') s2 = _("new      ");
    else if (file2->status == 'd') s2 = _("deleted  ");
    else if (file2->status == 'm') s2 = _("mode     ");
    else                           s2 = _("changed  ");

    path = file1 ? file1->path : file2->path;

    if (default_action == '/') s3 = "/";
    else if (default_action == '<') s3 = "<";
    else s3 = ">";

    printf("%s %s %s  %s  [%s] ", s1, oper, s2, path, s3);
    answer = get_keypress();

    if (answer == '\r' || answer == '\n') answer = default_action;

    /* If a choice was made, update the displayed line */
    if (answer == '<')      printf("\r%s <== %s  %s  [%s] ", s1, s2, path, s3);
    else if (answer == '>') printf("\r%s ==> %s  %s  [%s] ", s1, s2, path, s3);
    else if (answer == '/') printf("\r%s     %s  %s  [%s] ", s1, s2, path, s3);

    /* Echo the answer, end the line */
    if (signaled) printf("%s\n", _("(interrupted)"));
    else printf("%c\n", answer);

    if (answer == '?') {
      printf("\t%s\n", _("<return>  accept default action, if any"));
      printf("\t%s\n", _("   >      copy from 1 to 2"));
      printf("\t%s\n", _("   <      copy from 2 to 1"));
      printf("\t%s\n", _("   /      skip this pair"));
      printf("\t%s\n", _("   x      show file sizes and dates"));
      printf("\t%s\n", _("   p      back to previous pair"));
      printf("\t%s\n", _("   g      turn on batch mode (i.e., apply default actions)"));
      printf("\t%s\n", _("   q      quit, do not synchronize"));
    }
  }

  return answer;
}
