/*
  get_protocol() is called by start_on_server() to interpret the first
  line returned by a server. That line should be "ready 1" (where "1"
  is the highest protocol the server understands) or "ready 1 nocase"
  (if the server runs on a case-insensitive file system). It may also
  be a line "? nnn xxxx" (where nnn s a decimal number and xxxx a
  short text), if the server indicates that an error occurred and it
  couldn't start.
*/

#include "stdincls.h"
#include "types.e"
#include "getline.e"
#include "export.h"


/* get_protocol -- read the "ready" line from a server */
EXPORT bool get_protocol(FILE *in, char id[MACHINE_ID_LEN+1],
			 int *protocol, bool *case_sensitive)
{
  struct itimerval timer = {{0, 0}, {15, 0}};
  struct itimerval timer0 = {{0, 0}, {0, 0}};
  char *line, *p = NULL;
  int saved_errno;
  bool ok;

  /* Wait at most 15 seconds for input from the server. The first line
     from the server should be "ready N" or "ready N nocase", where N
     is the highest protocol version the server understands. "nocase"
     means the server runs on a file system that is case-insensitive.
  */
  (void) setitimer(ITIMER_REAL, &timer, NULL);
  errno = 0;
  ok = (line = getline_chomp(in));
  saved_errno = errno;
  (void) setitimer(ITIMER_REAL, &timer0, NULL);

  if (saved_errno == EINTR) {
    warnx(_("Timeout waiting for server"));
  } else if (saved_errno != 0) {
    warn(NULL);
  } else if (!ok) {
    warnx(_("No response from server"));
  } else if (strncmp(line, "ready ", 6) != 0) {
    warnx(_("Unrecognized reply from server: '%s'"), line); ok = false;
  } else if (strspn(line + 6, "0123456789abcdef") != MACHINE_ID_LEN) {
    warnx(_("Incorrect machine ID in reply from server: %s"), line+6); ok = false;
  } else if ((*protocol = strtol(line + 6 + MACHINE_ID_LEN, &p, 10)) <= 0) {
    warnx(_("Unrecognized protocol version from server: %s"), line+6); ok = false;
  } else {
    *case_sensitive = !strstr(p, "nocase");
  }

  if (ok) {
    strncpy(id, line + 6, MACHINE_ID_LEN);
    id[MACHINE_ID_LEN] = '\0';
  }
  return ok;
}


