/*
  digest() is called by generate_signature() and generate_delta() to
  compute a strong checksum of a block of bytes.

  Implementation in protocol 1:

  digest() uses MD5, which creates a 16-byte cryptographic
  checksum.
*/

#include "stdincls.h"
#include "types.e"		/* Defines DIGEST_LEN */
#ifndef USE_BLAKE2
# include "s-md5.e"
#endif

struct digest_state {
#ifdef USE_BLAKE2
  blake2b_state S[1];
#else
  MD5_CTX S[1];
#endif
};
EXPORT typedef struct digest_state *digest_state;


/* digest_init -- initialize a state for progressive computation of a digest */
EXPORT void digest_init(digest_state *state)
{
  *state = malloc(sizeof(**state));
#ifdef USE_BLAKE2
  blake2b_init((*state)->S, DIGEST_LEN);
#else
  MD5Init((*state)->S);
#endif
}


/* digest_update -- add a block to the digest */
EXPORT void digest_update(digest_state state, unsigned const char *buf,
			  const size_t len)
{
#ifdef USE_BLAKE2
  (void) blake2b_update(state->S, (uint8_t*)buf, len);
#else
  MD5Update(state->S, (unsigned char*)buf, len);
#endif
}


/* digest_final -- extract the digest from a state */
EXPORT void digest_final(digest_state state, unsigned char digest[DIGEST_LEN])
{
#ifdef USE_BLAKE2
  (void) blake2b_final(state->S, digest, DIGEST_LEN);
#else
  MD5Final(digest, state->S);
#endif
  free(state);
}


/* digest -- compute a strong hash of a circular block of bytes */
EXPORT void digest(unsigned const char *buf, size_t bufsize, size_t start,
		   size_t len, unsigned char digest[DIGEST_LEN])
{
  /* "Circular block" means that the data wraps around at the end of
     the block. I.e., the bytes are buf[start]...buf[bufsize-1],
     buf[0]...buf[start].
  */
#ifdef USE_BLAKE2
  blake2b_state S[1];
  int r;

  r = blake2b_init(S, DIGEST_LEN);
  assert(r != -1);
  if (start + len > bufsize) {
    r = blake2b_update(S, (uint8_t*)buf + start, bufsize - start);
    assert(r >= 0);
    r = blake2b_update(S, (uint8_t*)buf, start + len - bufsize);
    assert(r >= 0);
  } else {
    r = blake2b_update(S, (uint8_t*)buf + start, len);
    assert(r >= 0);
  }
  r = blake2b_final(S, digest, DIGEST_LEN);
  assert(r >= 0);
#else
  MD5_CTX S[1];

  MD5Init(S);
  if (start + len > bufsize) {
    MD5Update(S, (unsigned char*)buf + start, bufsize - start);
    MD5Update(S, (unsigned char*)buf, start + len - bufsize);
  } else {
    MD5Update(S, (unsigned char*)buf + start, len);
  }
  MD5Final(digest, S);
#endif
}
