/*
  Some shared types and constants.
*/

#include "stdincls.h"

#define R2SYNCPROTO 1		/* Highest known server protocol version */
EXPORTDEF(R2SYNCPROTO)

#define R2SYNCDIR ".r2sync"	/* Relative to $HOME */
EXPORTDEF(R2SYNCDIR)

#ifdef USE_BLAKE2
#define DIGEST_LEN 32		/* Bytes in Blake2 digest */
#else
#define DIGEST_LEN 16		/* Bytes in MD5 digest */
#endif
EXPORTDEF(DIGEST_LEN)

#define MACHINE_ID_LEN 32	/* 32-char, lowercase hex string */
EXPORTDEF(MACHINE_ID_LEN)

EXPORT typedef enum {
  IS_DIR, IS_FILE, IS_OTHER
} filetype;

EXPORT typedef struct {
  uint32_t sum;
  unsigned char digest[DIGEST_LEN];
} suminfo;

/*
  fileinfo is used both by the server part and the client part to
  store info about files. The "sums" field, for checksums and digests,
  is only used by the server part. The status field is only used by
  the client part.

  '=' = (client) the file is unchanged relative to the stored log.
  'n' = (client) the file is new, i.e., did not occur in the log.
  'd' = (client) the file is deleted, i.e., in the log, but not on disk.
  'm' = (client) the file's mode changed relative to the log.
  '-' = (server) the field "sums" is not in use.
  '+' = (server) the field "sums" is in use.
*/
EXPORT typedef struct {
  char status;
  unsigned int mode;
  time_t time;
  long long size;
  suminfo sums;
  char *path;
} fileinfo;

EXPORT int signaled = 0;	/* Flag set to signal # by signal handlers */

/* Conventional short macro for gettext() */
#define _(String) gettext(String)
#define gettext_noop(String) String
#define N_(String) gettext_noop(String)
EXPORTDEF(_(String))
EXPORTDEF(gettext_noop(String))
EXPORTDEF(N_(String))


/* cmp_fileinfo -- compare two fileinfo entries alphabetically */
EXPORT int cmp_fileinfo(const void *a, const void *b)
{
  return strcmp(((fileinfo*)a)->path, ((fileinfo*)b)->path);
}


/* cmpcase_fileinfo -- compare two fileinfo entries ignoring case */
EXPORT int cmpcase_fileinfo(const void *a, const void *b)
{
  return strcasecmp(((fileinfo*)a)->path, ((fileinfo*)b)->path);
}


/* create_directories -- create all directories leading up to path */
EXPORT bool create_directories(char *path)
{
  char *p = path;
  bool ok = true;

  /* Try to create all directories in path that don't exist yet. Only
     the components before the last '/' are created. Anything after
     the last '/' is ignored. The path may be relative or absolute
     (i.e., starting with a '/'). Two slashes in a row (e.g.,
     "dir//subdir" are treated as a single.
  */
  assert(path);
  do {
    while (*p == '/') p++;
    p = strchr(p, '/');
    if (p) {
      *p = '\0';		/* Temporarily replace '/' by '\0' */
      ok = mkdir(path, 0777) == 0 || errno == EEXIST;
      *p = '/';			/* Restore '/' */
    }
  } while (ok && p);

  return ok;
}


/* debug -- print debugging information */
EXPORT void debug(const char *format,...)
{
#ifdef DEBUG
  va_list ap;
  int pid, saved_errno;

  saved_errno = errno;
  pid = getpid();
  /* Ignore the fact that flock() fails if the file descriptor is a pipe.:-( */
  flock(2, LOCK_EX);
  fprintf(stderr, "[%d] ", pid);
  va_start(ap, format);
  vfprintf(stderr, format, ap);
  va_end(ap);
  flock(2, LOCK_UN);
  errno = saved_errno;
#endif /* DEBUG */
}


/* make_absolute -- concatenate a base and a relative path, false if too long */
EXPORT bool make_absolute(const char *base, const char *path,
			  char abspath[FILENAME_MAX+1])
{
  size_t b = strlen(base);
  size_t p = strlen(path);

  if (b + p > FILENAME_MAX) return false;
  memcpy(abspath, base, b);
  memcpy(abspath + b, path, p);
  abspath[b+p] = '\0';
  return true;
}


#define hexval(c) ('0' <= (c) && (c) <= '9' ? (c) - '0' :	\
		   'a' <= (c) && (c) <= 'f' ? 10 + (c) - 'a' :	\
		   'A' <= (c) && (c) <= 'F' ? 10 + (c) - 'A' :	\
		   16)


/* str_to_digest -- parse a hex-encoded digest */
EXPORT bool str_to_digest(char * restrict s, unsigned char d[DIGEST_LEN],
			  char ** restrict end)
{
  uint32_t h;
  size_t j;

  for (j = 0; j < 2 * DIGEST_LEN && (h = hexval(*s)) != 16; s++, j++)
    if (j % 2 == 0) d[j/2] = h << 4; /* 4 high bits of the byte */
    else d[j/2] |= h;		     /* 4 low bits of the byte */

  *end = s;
  return j == 2 * DIGEST_LEN;
}


/* digest_to_str -- return a static string with the digest in hexadecimal */
EXPORT char *digest_to_str(const unsigned char m[DIGEST_LEN])
{
  static const char hexchar[] = "0123456789abcdef";
  static char hex[2*DIGEST_LEN+1];
  int i, j;

  for (i = j = 0; i < DIGEST_LEN; i++) {
    hex[j++] = hexchar[m[i] >> 4];
    hex[j++] = hexchar[m[i] & 0x0F];
  }
  hex[j] = '\0';
  return hex;
}
