/*
  get_file_lists() is called by client() to get the list of changed
  files (with their mode, time and size) from both servers.

  Implementation:

  get_file_lists() sends both servers a "list" command and then reads
  the list of files returned, one file per line. A line consisting of
  a single dot signals the end of the list.
*/

#include "stdincls.h"
#include "types.e"
#include "getline.e"		/* Defines getline_chomp() */
#include "print.e"		/* Defines print() */
#include "c-profile.e"		/* Defines struct Profile */


/* get_one_list -- read the list sent on stream f, store it in *list */
static bool get_one_list(int rootnumber, FILE *f, fileinfo **list, int *n)
{
  char *line, c;
  bool ok = true;	 /* Set to false after we reported an error */
  int i;

  /* Read lines until a line with a dot or an error */
  while ((line = getline_chomp(f)) && strcmp(line, ".") && line[0] != '?')
    if (ok) {
      if (!(*list = realloc(*list, (*n + 1) * sizeof(**list)))) {
	warn(NULL); ok = false;
      } else if (sscanf(line, "%c %o %ld %lld%c%n", &(*list)[*n].status,
			&(*list)[*n].mode, &(*list)[*n].time, &(*list)[*n].size,
			&c, &i) != 5 || c != ' ') {
	warnx(_("Syntax error in response from server")); ok = false;
      } else if (!((*list)[*n].path = strdup(line + i))) {
	warn(NULL); ok = false;
      } else {
	(*n)++;
      }
    }

  /* Check if the server sent an error */
  if (ferror(f)) {
    warn(_("Error on root %d"), rootnumber);
    ok = false;
  } else if (!line) {
    warnx(_("Unexplained error on root %d"), rootnumber);
    ok = false;
  } else if (line[0] == '?') {
    warnx(_("Error on root %1$d: %2$s"), rootnumber, line + 2);
    ok = false;
  }

  return ok && !signaled;	/* False if a signal handler set a flag. */
}


/* get_file_lists -- get the lists of files and their status from each server */
EXPORT bool get_file_lists(Profile profile,
			   fileinfo **files1, int *nfiles1,
			   fileinfo **files2, int *nfiles2)
{
  bool ok1 = true, ok2 = true;
  char *line;

  /* Instruct servers to make lists of changed files */
  print1(profile->to1, "list\n");
  print1(profile->to2, "list\n");

  /* Inform the user if there are no logs of a previous synchronization */
  if (!(line = getline_chomp(profile->from1))) {
    warn(_("Error on root 1"));
    ok1 = false;
  } else if (line[0] == '?') {
    warnx(_("Error on root 1: %s"), line + 2);
    ok1 = false;
  } else if (strcmp(line, "creating") == 0)
    printf(_("No log found on root 1. Treating all files as new.\n"));

  if (!(line = getline_chomp(profile->from2))) {
    ok2 = false; warn(_("Error on root 2"));
  } else if (line[0] == '?') {
    ok2 = false; warnx(_("Error on root 2: %s"), line + 2);
  } else if (strcmp(line, "creating") == 0)
    printf(_("No log found on root 2. Treating all files as new.\n"));

  /* Read the lists of files sent by both servers */
  if ((ok1 || ok2) && !profile->quiet) printf(_("Waiting for file lists...\n"));
  if (ok1) ok1 = get_one_list(1, profile->from1, files1, nfiles1);
  if (ok2) ok2 = get_one_list(2, profile->from2, files2, nfiles2);

  return ok1 && ok2 && !signaled; /* False if a signal handler set a flag. */
}
