/*
  compute_delta() is called by serve1() to compute a series of patches
  given an up-to-date file and the signature of an outdated file.

  Implementation in protocol 1:

  compute_delta() first sends a line with the file's checksum and
  digest, unless "noshortcuts" was passed in the "version" command.

  It then reads a signature (a series of checksums) in the format
  output by compute_signature() (see file "s-signature.c") and uses a
  hash table to store the checksums and block numbers.

  It then reads the file given by path and sends the client a delta,
  which is an alternating sequence of literal bytes (in base64) and
  references to blocks defined in the signature. See the description
  in protocol.html#update for the exact format.
*/

#include "stdincls.h"
#include "types.e"
#include "getline.e"
#include "print.e"
#include "errcodes.e"
#include "s-hash.e"
#include "s-checksum.e"
#include "s-digest.e"
#include "s-sumfile.e"
#include "s-store.e"

/*
  The checksum of a block and its size are the keys into a hash
  table. The digest and the block number are the data. There can be
  multiple data items for the same checksum.
*/
struct blockkey {
  uint32_t sum;
  unsigned long blocksize;
};
struct blockinfo {
  unsigned long blocknr;
  unsigned char digest[DIGEST_LEN];
};

/*
  A structure to help with the run-length encoding of the output.
  in_blocks is true if we're currently printing block numbers instead
  of literal bytes. If so, block, block+1,... block+count are still to
  be printed. len is the length of the line in characters printed so
  far, so we can insert newlines every 72 characters or so.

  The routine print_block() doesn't immediately print the block
  number, but stores it in the printing_state.

  Likewise, print_byte() accumulates bytes in the printing_state. They
  are printed (in base64) only qwhen the line becomes longer than
  LINELEN, or the next thing to add is not a byte, but a block number.
*/
#define LINELEN 8000
typedef struct _printing_state {
  FILE *f;			/* The stream to print to */
  bool in_blocks;		/* Whether we're outputting blocks or bytes */
  unsigned long count, block;	/* block...(block+count) still to print */
  int len;			/* Length of line printed so far */
  char buf[LINELEN+2];		/* Collect base64 characters + \n + \0 */
  unsigned char bits;		/* Store base64 bits between calls */
} printing_state;

static char base64[] =
  "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";


/* flush_blocknumbers -- print unprinted block number and count */
static void flush_blocknumbers(printing_state *state)
{
  assert(state->in_blocks);
  if (state->len == 0) state->len = print(state->f, "*%lu", state->block);
  else state->len += print(state->f, " %lu", state->block);
  if (state->count != 0) state->len += print(state->f, "+%lu", state->count);
  if (state->len >= LINELEN) {print1(state->f, "\n"); state->len = 0;}
}


/* print_byte_line -- print the accumulated base64 line */
static void print_byte_line(printing_state *state)
{
  if (state->len & 0x3) state->buf[state->len++] = base64[state->bits];
  if (state->len & 0x3) state->buf[state->len++] = '=';
  if (state->len & 0x3) state->buf[state->len++] = '=';
  state->buf[state->len] = '\n';
  state->buf[state->len+1] = '\0';
  print1(state->f, state->buf);
  state->len = 0;
}


/* print_byte -- print any unprinted block numbers and add a byte in base64 */
static void print_byte(printing_state *state, unsigned char b)
{
  if (state->in_blocks) {
    flush_blocknumbers(state);
    if (state->len != 0) {print1(state->f, "\n"); state->len = 0;}
    state->in_blocks = false;
  } else if (state->len >= LINELEN) {
    print_byte_line(state);
  }
  switch (state->len & 0x3) {
  case 0:
    state->buf[state->len++] = base64[b >> 2]; /* Use 6 bits */
    state->bits = (b & 0x3) << 4;	       /* Keep 2 bits */
    break;
  case 1:
    state->buf[state->len++] = base64[state->bits | b >> 4]; /* Use 4 bits */
    state->bits = (b & 0xF) << 2; /* Keep 4 bits */
    break;
  case 2:
    state->buf[state->len++] = base64[state->bits | b >> 6]; /* Use 2 bits*/
    state->buf[state->len++] = base64[b & 0x3F]; /* Use 6 bits */
    break;
  case 3:
    assert(!"Cannot happen!");
  }
}


/* print_block -- add a blocknumber to the output */
static void print_block(printing_state *state, unsigned long n)
{
  if (!state->in_blocks) {
    if (state->len != 0) print_byte_line(state); /* End the line of bytes */
    state->in_blocks = true;
    state->block = n;
    state->count = 0;
  } else if (n == state->block + state->count + 1) { /* Next in sequence */
    state->count++;
  } else {
    flush_blocknumbers(state);
    state->block = n;
    state->count = 0;
  }
}


/* print_end -- print unprinted block numbers, print a newline */
static void print_end(printing_state *state)
{
  if (state->in_blocks) {
    flush_blocknumbers(state);
    if (state->len != 0) {print1(state->f, "\n"); state->len = 0;}
  } else if (state->len != 0)
    print_byte_line(state);
}


/* key_hash -- compute a hash for a blockkey struct */
static unsigned long key_hash(const void *key, size_t keylen)
{
  return ((struct blockkey*)key)->sum;
}


/* key_cmp -- check if two blockkeys are the same */
static bool key_cmp(const void *a, const size_t alen,
		    const void *b, const size_t blen)
{
  return ((struct blockkey*)a)->sum == ((struct blockkey*)b)->sum &&
    ((struct blockkey*)a)->blocksize == ((struct blockkey*)b)->blocksize;
}


/* send_sums -- send checksum & digest (path is abs., file relative to root) */
static bool send_sums(FILE *out, const char *path,
		      const char* file, const DB store)
{
  fileinfo info;

  /* Check if, by chance, the sum and digest were already computed
     (most likely in reply to an earlier "sums" command). Otherwise
     compute them. */
  if (store_get(store, file, &info) && info.status == '+')
    debug("Found checksum for %s in store\n", path);
  else if (file_checksum_and_digest(path, &info.sums.sum, info.sums.digest))
    debug("Computed checksum (%x) and digest for %s\n", info.sums.sum, path);
  else {
    print(out, "? %03d %s\n", EC_SERVER + errno, strerror(errno));
    return false;
  }

  return
    print(out, "%x %s\n", info.sums.sum, digest_to_str(info.sums.digest)) > 0 &&
    !signaled;
}


/* read_signature -- read the list of checksums, store them in the hash table */
static bool read_signature(FILE *in, FILE *out, hashtable table)
{
  unsigned long blocknumber = 1; /* First block has number 1 */
  bool err = false, result;
  char *line, *p;
  struct blockkey key;
  struct blockinfo info;

  /* Read the checksums for block blocknumber and store them in a hash table */
  while ((line = getline1(in)) && line[0] != '.' && line[0] != '?') {

    /* A checksum, a digest and a blocksize, all in hex, separated by spaces */
    key.sum = strtoul(line, &p, 16);
    info.blocknr = blocknumber;
    if (!str_to_digest(p+1, info.digest, &p) ||
	(key.blocksize = strtoul(p+1, &p, 10)) == 0)
      if (!err) {print(out, "? %03d Syntax error\n", EC_SYNTAX); err = true;}

    debug("%x %s %lx\n", key.sum, digest_to_str(info.digest), key.blocksize);

    /* Add the sum, digest and size to the hash table */
    if (!err && !hash_add(table, &key, sizeof(key), &info, sizeof(info))) {
      print(out, "? %03d %s\n", EC_SERVER + errno, strerror(errno));
      err = true;
    }

    blocknumber++;
  }

  result = !err && line && line[0] != '?';
  return result;
}


/* generate_delta -- read the file given by path and output a delta */
static void generate_delta(FILE *out, const char *path, unsigned long blocksize,
			   hashtable table)
{
  unsigned char buf[blocksize], d[DIGEST_LEN];
  size_t bufindex = 0;
  printing_state state;
  unsigned long blocknumber;
  bool found;
  int c, i;
  FILE *f;
  struct blockkey key;
  struct blockinfo *info;
  size_t unused;

  /* Open the file and lock it while we're using it */
  if (!(f = fopen(path, "rb")) || flock(fileno(f), LOCK_EX) == -1) {
    print(out, "? %03d %s\n", EC_SERVER + errno, strerror(errno));
    return;
  }

  /* Use the rolling checksum to find blocks that are potentially available */
  state.in_blocks = false;
  state.f = out;
  state.len = 0;
  key.blocksize = fread(buf, 1, blocksize, f);
  if (key.blocksize) key.sum = checksum(buf, blocksize, 0, key.blocksize);
  while (key.blocksize) {	/* While still bytes in buf */

    /* Check if we have one or more matches for this block. */
    found = false;
    for (i = 1; (info = hash_nget(table, &key, sizeof(key), i, &unused)); i++) {

      /* We have a match for the checksum and the blocksize, now check
	 the strong hash. (Compute it only once for this block.)
       */
      if (i == 1) digest(buf, blocksize, bufindex, key.blocksize, d);

      if (memcmp(info->digest, d, DIGEST_LEN) == 0) {
	/* Found a match for the strong hash as well. If this is the
	   first matching block, or it is better than the previous
	   one, store the number in blocknumber. A number is "better"
	   if it allows the run-length encoding. (We're not going as
	   far as trying a better previous block... ) */
	if (!found || info->blocknr == state.block + state.count + 1)
	  blocknumber = info->blocknr;
	found = true;
      }
    }

    if (found) {		/* We found a matching block */
      print_block(&state, blocknumber);	/* Output a reference to it */

      key.blocksize = fread(buf, 1, blocksize, f); /* Read the next block */
      bufindex = 0;
      if (key.blocksize)
	key.sum = checksum(buf, blocksize, bufindex, key.blocksize);

    } else {			/* We didn't find a matching block */
      print_byte(&state, buf[bufindex]); /* Output the 1st byte of the block */
      key.sum = checksum_remove_one(key.sum, buf[bufindex], key.blocksize);	/* Remove 1st byte */

      if ((c = getc(f)) == EOF) { /* Try to read the next byte from f */
	key.blocksize--;     /* No new byte, make the block shorter */
      } else {
	buf[bufindex] = c;	/* Add to the circular buffer */
	key.sum = checksum_add_one(key.sum, c); /* Update checksum */
      }
      bufindex = (bufindex + 1) % blocksize; /* New start of circular buffer */
    }
  }

  print_end(&state);
  assert(state.len == 0);
  if (feof(f))
    print1(out, ".\n");
  else
    print(out, "? %03d %s\n", EC_SERVER + errno, strerror(errno));

  fclose(f);
}


/* compute_delta -- compute the delta given a file and a signature */
EXPORT void compute_delta(FILE *in, FILE *out, const char *root,
			  const unsigned long blocksize, const char *file,
			  const bool use_shortcut, const DB store)
{
  char path[FILENAME_MAX+1];
  hashtable table;

  assert(root);
  assert(file);

  debug("Start compute_delta()\n");

  if (!make_absolute(root, file, path)) {
    print(out, "? %03d Path too long\n", EC_TOOLONG);
    return;
  }
  if (!(table = hash_create(key_hash, key_cmp))) {
    print(out, "? %03d %s\n", EC_SERVER + errno, strerror(errno));
    return;
  }
  if (!use_shortcut)
    print(out, "OK\n");
  else if (!send_sums(out, path, file, store))
    return;

  assert(*path);

  if (read_signature(in, out, table))
    generate_delta(out, path, blocksize, table);

  hash_destroy(table);
}
