/*
 * list() is called by serve1() after it receives a "list" command. It
 * compares the existing files under "root" to the logs kept in
 * "store" and prints a list of changed files, with their mode,
 * modification time and size.
 *
 * In protocol version 1, the printed lines have the form
 *
 *     S MODE TIME SIZE PATH
 *
 * PATH is the path of a file (in UTF-8), relative to the "root".
 *
 * SIZE is the size of the file in bytes (as a decimal number). If the
 * file doesn't exist (i.e., when S = d), this number is ignored, but
 * should be 0.
 *
 * TIME is the modification time of the file, in seconds since the
 * Unix epoch (0:00 UTC on 1 Jan 1970), in decimal. If the file
 * doesn't exist (i.e., when S = d), this number is ignored.
 *
 * MODE is the mode of the file, as given by "st_mode" in "struct
 * stat", see lstat(2). This number is printed in octal.
 *
 * S is a single letter "n", "d" or "u". "n" means the file is new
 * (i.e., does not occur in the logs). "d" means the file is deleted
 * (i.e., it occurs in the logs, but cannot be found under "root").
 * And "u" means the file is updated (i.e., the mode, time or size are
 * different in the logs than for the actual file).
 *
 * The list ends with a line with a single dot:
 *
 *     .
 *
 * If an error occurs, list() prints a line of the form
 *
 *     ? DDD TEXT
 *
 * where DDD is a decimal number and TEXT is a short explanation
 * (e.g., Out of memory) and prints no more lines. No line with a dot
 * is printed either.
 */

#include "stdincls.h"
#include "types.e"
#include "s-digest.e"
#include "errcodes.e"
#include "s-sumfile.e"
#include "print.e"
#include "s-store.e"


/* print_info -- print a line with file info */
static void print_info(FILE *out, char status, fileinfo info)
{
  assert(info.path);
  debug("send: %c %o %ld %lld %s\n", status, info.mode, info.time, info.size,
	info.path);
  print(out, "%c %o %ld %lld %s\n", status, info.mode, info.time, info.size,
	info.path);
}


/* find_files -- get the list of all files under root, -1 if error */
static int find_files(char * const root, fileinfo **files)
{
  char searchroot[PATH_MAX+1];
  char *roots[2] = {searchroot, NULL};
  int n = 0;
  FTSENT *e;
  FTS *fts;
  size_t prefixlen;
  bool ok = true;

  debug("starting fts\n");

  strcpy(searchroot, root);
  prefixlen = strlen(root);
  if (prefixlen > 1 && root[prefixlen-1] == '/') searchroot[prefixlen-1] = '\0';

  /* Walk the tree under root */
  *files = NULL;
  if (!(fts = fts_open(roots, FTS_COMFOLLOW | FTS_PHYSICAL, NULL))) return -1;
  while (ok && (e = fts_read(fts))) {

    /* We're only interested in regular files and symbolic links */
    if (e->fts_info == FTS_F || e->fts_info == FTS_SL) {
      if (!(*files = realloc(*files, (n + 1) * sizeof(**files))))
	ok = false;
      else if (!((*files)[n].path = strdup(e->fts_path + prefixlen)))
	ok = false;
      else {
	(*files)[n].status = '-'; /* Indicate that "sums" is empty */
	(*files)[n].mode = e->fts_statp->st_mode;
	(*files)[n].time = e->fts_statp->st_mtime;
	(*files)[n].size = e->fts_statp->st_size;
	n++;
      }
    }
  }

  if (fts_close(fts) == -1) ok = false;

  /* Free memory if an error occurred */
  if (!ok && *files) {
    for (n--; n >= 0; n--) free((*files)[n].path);
    free(*files);
  }

  return ok ? n : -1;
}


/* get_files_from_store -- get all files from the log, or -1 if error */
static int get_files_from_store(DB store, fileinfo **files)
{
  DB_cursor cursor;
  int n = 0;

  *files = NULL;

  /* Get all files from the store, if any */
  if (!(cursor = cursor_create(store))) return -1;
  while ((*files = realloc(*files, (n + 1) * sizeof(**files))) &&
	 cursor_next(cursor, &(*files)[n])) n++;

  return *files ? n : -1;
}


/* list -- print the list of changed, new or deleted files */
EXPORT void list(FILE *out, DB store, char * const root, bool case_sensitive)
{
  fileinfo *oldfiles = NULL, *curfiles = NULL;
  int ncurfiles = 0, noldfiles = 0, r, i, j;
  bool ok = true;

  assert(root && root[0]);

  /* Collect all files under root into curfiles, sort them */
  if ((ncurfiles = find_files(root, &curfiles)) == -1) {
    print(out, "? %03d %s\n", EC_SERVER + errno, strerror(errno));
    return;
  }
  qsort(curfiles, ncurfiles, sizeof(*curfiles),
	case_sensitive ? cmp_fileinfo : cmpcase_fileinfo);

  debug("found %d files in directory %s\n", ncurfiles, root);

  /* Collect all files from the log */
  if ((noldfiles = get_files_from_store(store, &oldfiles)) == -1) {
    print(out, "? %03d %s\n", EC_SERVER + errno, strerror(errno));
    ok = false;
  } else
    qsort(oldfiles, noldfiles, sizeof(*oldfiles),
	  case_sensitive ? cmp_fileinfo : cmpcase_fileinfo);

  debug("found %d files in the log\n", noldfiles);

  /* Find files that are new, changed or deleted */
  for (i = j = 0; ok && (i < ncurfiles || j < noldfiles);) {

    if (i == ncurfiles) r = 1;
    else if (j == noldfiles) r = -1;
    else if (case_sensitive) r = cmp_fileinfo(&curfiles[i], &oldfiles[j]);
    else r = cmpcase_fileinfo(&curfiles[i], &oldfiles[j]);

    if (r < 0) {		/* curfiles[i] does not occur in oldfiles */
      print_info(out, 'n', curfiles[i]);
      i++;
    } else if (r > 0) {	/* oldfiles[j] does not occur in curfiles */
      print_info(out, 'd', oldfiles[j]);
      j++;
    } else if (curfiles[i].time != oldfiles[j].time ||
	       curfiles[i].size != oldfiles[j].size) { /* File modified */
      print_info(out, 'u', curfiles[i]);
      i++;
      j++;
    } else if (curfiles[i].mode != oldfiles[j].mode) { /* File mode modified */
      print_info(out, 'm', curfiles[i]);
      i++;
      j++;
    } else {			/* File was not modified */
      print_info(out, '=', oldfiles[j]);
      i++;
      j++;
    }
  }
  if (ok) print1(out, ".\n");
  else print(out, "? %03d %s\n", EC_SERVER + errno, strerror(errno));

  /* Free allocated memory */
  while (ncurfiles) free(curfiles[--ncurfiles].path);
  free(curfiles);

  /* Free the array. Do not free the paths inside oldfiles, because
     they are still used by the store */
  free(oldfiles);
}
