/*
  Check if a file on root1 and a file on root2 are the same: same
  time, size and checksums.

  same_file() is called by determine_file_actions() when both roots
  report that the file changed and the option auto-check is on. It
  checks first if the time and size are the same. If not, it reports
  that the files are not the same. Otherwise, it asks the servers to
  compute a checksum and a digest of the file. It then reports whether
  those sums and digests are are same.

  Asking the servers is only possible in protocol 2 or later.
*/

#include "stdincls.h"
#include "types.e"		/* Defines struct fileinfo */
#include "print.e"		/* Defines print() */
#include "getline.e"		/* Defines getline_chomp() */
#include "c-profile.e"		/* Defines Profile */


/* read_sum_and_digest -- get the sum and digest for a file from server root */
static bool read_sum_and_digest(FILE *from, const char *path, int root,
  suminfo *sums)
{
  char *p, *line;

  if (!(line = getline_chomp(from))) {
    warn(_("Error on root %d"), root);
    return false;
  } else if (line[0] == '?') {
    warnx(_("Error on root %1$d: %2$s"), root, line + 2);
    return false;
  } else if ((sums->sum = strtoul(line, &p, 16)),
    (*p != ' ' || !str_to_digest(p + 1, sums->digest, &p))) {
    warnx(_("Incorrect response from root %1$d: %2$s"), root, line);
    return false;
  }
  return true;
}


/* same_file -- check that files have the same time, size and checksums */
EXPORT bool same_file(Profile profile, fileinfo *file1, fileinfo *file2)
{
  suminfo c1, c2;

  debug("autocheck for %s and %s\n", file1->path, file2->path);
  return profile->protocol >= 2 && /* Only possible in protocol 2 or newer */
    file1->time == file2->time &&
    file1->size == file2->size &&
    print(profile->to1, "sums %s\n", file1->path) > 0 &&
    print(profile->to2, "sums %s\n", file2->path) > 0 &&
    read_sum_and_digest(profile->from1, file1->path, 1, &c1) &&
    read_sum_and_digest(profile->from2, file2->path, 2, &c2) &&
    c1.sum == c2.sum &&
    memcmp(c1.digest, c2.digest, DIGEST_LEN) == 0;
}
