/*
  synchronize() is called by client() to update the given list of
  files on a server. The streams toA and fromA are used to talk to
  that server. toB and fromB are used to talk to the other server,
  which is the server that has the up to date files.

  Implementation using protocol version 1:

  For each file in the list, synchronize() sends an "update" command
  to the first server and a "delta" command to the other server, and
  then copies the signature (a series of checksums for each block in
  the file) it receives from the first server to the second server.
  Once the second server has the whole signature, it computes and
  returns a delta (series of patches). synchronize() then copies all
  the patches to the first server.

  synchronize() selects a block size depending on the file size: the
  larger the file, the larger the block size, on the assumption that
  large files will have larger parts that are unchanged, and sending
  large blocks is faster (less overhead) than sending small blocks.
  Following rsync's example, the block size is approximately the
  square root of the file size (with a minimum of

  synchronize() doesn't interpret the received signature or patches,
  but only copies them from one server to the other, until a line with
  one dot that signals the end of the signature or of the delta.

  Both servers can send error message, which abort the operation.
  synchronize() reports them to the user.

  One error message is treated specially: error code 200 (EC_SHORTCUT)
  means the server succeeded in updating a file without requiring a
  delta from the other server, because it found an identical local
  file and copied that. (It is treated as an error in the sense that
  it halts the operation: no signature will be sent and no delta sent
  back. But it is also a success, in the sense that a file was
  successfully updated.)
*/

#include "stdincls.h"
#include "types.e"
#include "errcodes.e"
#include "getline.e"
#include "print.e"
#include "c-profile.e"
#include "c-progress.e"

#define DEFAULT_BLOCK 1024	/* Default block size for checksums */


/* update_file -- pass signature from A to B and delta from B to A */
static bool update_file(FILE *toA, FILE *fromA, FILE *toB, FILE *fromB,
			const int blocksize, const fileinfo file,
			const bool use_shortcut, bool *shortcut)
{
  char *line;

  assert(file.path);
  debug("sending to %d: delta %d %s\n", fileno(toB), blocksize, file.path);

  /* Issue a "delta" command to server B. */
  if (print(toB, "delta %d %s\n", blocksize, file.path) < 0) {
    warn(_("Failed to send command"));
    return false;
  }

  /* If we are using shortcuts, the response should be the file's
     checksum and digest, otherwise it should be the word "OK". */
  if (!(line = getline_chomp(fromB))) {
    warn(_("Failed to get response to delta command for %s"), file.path);
    return false;
  } else if (line[0] == '?') {
    warnx(_("Failed to get delta of %1$s: %2$s"), file.path, line + 2);
    return false;
  } else if (!use_shortcut && strcmp(line, "OK") != 0) {
    warnx(_("Unrecognized response to delta command for %1$s: \"%2$s\""),
	  file.path, line);
    return false;
  }

  /* Issue an "update" command to server A, including the checksum & digest. */
  if (use_shortcut ?
      print(toA, "update %d %o %ld %lld %s %s\n", blocksize, file.mode & 07777,
	    file.time, file.size, line, file.path) < 0 :
      print(toA, "update0 %d %o %ld %lld %s\n", blocksize, file.mode & 07777,
	    file.time, file.size, file.path) < 0) {
    warn(_("Failed to update %s"), file.path);
    return false;
  }

  /* Copy signature from server A to server B, until a "." or an error */
  do {
    if (!(line = getline1(fromA))) {
      warn(_("Failed to update %s"), file.path);
      (void) print(toB, "? %03d Incomplete signature\n", EC_NOSIG);
      return false;
    } else
      (void) print1(toB, line);
  } while (line[0] != '.' && line[0] != '?');

  /* If A sent EC_SHORTCUT, we're done; if it sent another error, we failed. */
  if (line[0] != '?') *shortcut = false;
  else if (atoi(line + 2) == EC_SHORTCUT) {*shortcut = true; return true;}
  else {warnx(_("Failed to update %1$s: %2$s"), file.path, line+2); return false;}

  /* Copy delta from server B to server A */
  do {
    if (!(line = getline1(fromB))) {
      warn(_("Failed to update %s"), file.path);
      (void) print(toA, "? %03d Incomplete delta\n", EC_NODELTA);
      return false;
    } else
      print1(toA, line);
  } while (line[0] != '.' && line[0] != '?');

  /* If server B sent an error, we failed */
  if (line[0] == '?') {
    warnx(_("Failed to update %1$s: %2$s"), file.path, line + 2);
    return false;
  }

  /* We copied without problems, wait for A to report success or failure */
  if (!(line = getline1(fromA))) {
    warn(_("Failed to update %s"), file.path);
    return false;
  } else if (line[0] == '?') {
    warnx(_("Failed to update %1$s: %2$s"), file.path, line + 2);
    return false;
  } else
    return true;
}


/* isqrt -- compute the integer square root of num */
static long isqrt(long num)
{
  long res = 0;
  long bit = 1;

  /* "bit" starts at the highest power of four <= the argument. */
  while (bit <= LONG_MAX >> 2 && bit << 2 <= num)
    bit <<= 2;

  while (bit != 0) {
    if (num >= res + bit) {
      num -= res + bit;
      res = (res >> 1) + bit;
    } else
      res >>= 1;
    bit >>= 2;
  }
  return res;
}


/* synchronize1 -- update files on server A */
static bool synchronize1(Profile profile, FILE *toA, FILE *fromA, FILE *toB,
			 FILE *fromB, fileinfo *files, int nfiles, int dir)
{
  static char * const flag[2][3] = {{"<-", "<=", "<*"}, {"->", "=>", "*>"}};
  bool ok = true, shortcut;
  long bsize;
  char *line = NULL;
  int i, nerrors = 0;

  for (i = 0; i < nfiles && !signaled && (ok || profile->keep_going); i++) {

    shortcut = false;

    /* Inform the user */
    if (!profile->quiet && *files[i].path) /* Only if file name isn't empty */
      status(flag[dir][0], files[i].path, progress_meter(profile, files[i].size));

    if (S_ISLNK(files[i].mode))
      /* It's a symlink that changed. */
      ok =
	print(toB, "readlink %s\n", files[i].path) > 0 &&
	(line = getline_chomp(fromB)) && line[0] != '?' &&
	print(toA, "symlink %ld %s\n", files[i].time, files[i].path) > 0 &&
	print(toA, "%s\n", line + 2) > 0 && /* Skip initial "= " */
	(line = getline1(fromA)) && line[0] != '?';
    else if (files[i].status == 'm')
      /* Update only the mode bits. */
      ok =
	print(toA, "chmod %o %s\n", files[i].mode & 07777, files[i].path) > 0 &&
	(line = getline1(fromA)) && line[0] != '?';
    else {
      /* Copy a signature from A to B and then a delta from B to A */
      /* block size is sqrt(filesize) rounded down to multiple of 8 */
      if (DEFAULT_BLOCK * DEFAULT_BLOCK >= files[i].size) bsize = DEFAULT_BLOCK;
      else bsize = isqrt(files[i].size) & ~07l;
      ok = update_file(toA, fromA, toB, fromB, bsize, files[i],
		       profile->use_shortcuts, &shortcut);
    }

    /* Tell server B to update its log, too, now that A has the same file. */
    if (ok) {
      if (print(toB, "log %o %ld %lld %s\n", files[i].mode, files[i].time,
		files[i].size, files[i].path) <= 0 || !(line = getline1(fromB)))
	{ok = false; warn(_("Error updating %s"), files[i].path);}
      else if (line[0] == '?')
	{ok = false; warnx(_("Error updating %1$s: %2$s"), files[i].path, line+2);}
    }

    if (signaled) ok = false;

    /* Inform the user */
    if (!profile->quiet && *files[i].path) {
      if (ok) {putchar('\r'); status(flag[dir][1+shortcut], files[i].path, "");}
      putchar('\n');
    }
#if 0
    if (!ok && line && line[0] == '?')
      warnx(_("Error updating %1$s: %2$s"), files[i].path, line + 2);
    else if (!ok && errno) warn(_("Error updating %s"), files[i].path);
    else if (!ok) warnx(_("Error updating %s"), files[i].path);
#endif

    if (!ok) nerrors++;
  }

  return nerrors == 0 && !signaled;
}


/* synchronize -- update files on both servers  */
EXPORT bool synchronize(Profile profile, fileinfo *send_to1, int nsend1,
			fileinfo *send_to2, int nsend2)
{
  bool ok;

  ok = synchronize1(profile, profile->to1, profile->from1, profile->to2,
		    profile->from2, send_to1, nsend1, 0);
  if (ok || profile->keep_going)
    ok &= synchronize1(profile, profile->to2, profile->from2, profile->to1,
		       profile->from1, send_to2, nsend2, 1);
  return ok;
}
