/*
  getline1() reads a line up to and including any newline character
  and stores the total number of bytes read for statistical
  purposes. It returns NULL in case of an error or end of file,
  otherwise it returns a pointer to a static buffer with the line that
  was read. (The next getline1() or getline_chomp() will override that
  buffer.)

  getline_chomp() does the same, except that it removes the newline
  character from the end of the line.

  get_bytes_received() returns the total number of bytes read by
  getline_chomp() and getline1() from a given file descriptor (for
  file descriptors < 10).
*/

#include "stdincls.h"
#include "types.e"

/* The array is only for statistical purposes */
#define MAX_FDS 10
static long long bytes_received[MAX_FDS] = {0,0,0,0,0,0,0,0,0,0};


/* The static buffer used to store a line that was read */
static char *linebuf = NULL;
static size_t linesize = 0;


/* getline_chomp -- read a line with getline(3), remove any final \r & \n */
EXPORT char *getline_chomp(FILE *stream)
{
  ssize_t len;
  int fd = fileno(stream);

  if ((len = getline(&linebuf, &linesize, stream)) == -1) return NULL;
  if (fd < MAX_FDS) bytes_received[fd] += len;
  if (len > 0 && linebuf[len-1] == '\n') linebuf[--len] = '\0';
  if (len > 0 && linebuf[len-1] == '\r') linebuf[--len] = '\0';
  debug("read fd=%d \"%s\"\n", fd, linebuf);
  return linebuf;
}


/* getline1 -- read a line with getline(3) */
EXPORT char *getline1(FILE *stream)
{
  ssize_t len;
  int fd = fileno(stream);

  if ((len = getline(&linebuf, &linesize, stream)) == -1) return NULL;
  if (fd < MAX_FDS) bytes_received[fd] += len;
  debug("read fd=%d \"%s\"\n", fd, linebuf);
  return linebuf;
}


/* get_bytes_received -- return the # of bytes read on a file descriptor */
EXPORT long long get_bytes_received(int fd)
{
  return fd < MAX_FDS ? bytes_received[fd] : -1;
}
