/*
  file_checksum_and_digest() is called by compute_delta() and by sums().
  It computes the checksum and the digest for a whole file.
*/

#include "stdincls.h"
#include "types.e"
#include "s-digest.e"
#include "s-checksum.e"


/* file_checksum_and_digest -- compute checksum and digest for a whole file */
EXPORT bool file_checksum_and_digest(const char *path, uint32_t *checksum,
				     unsigned char digest[DIGEST_LEN])
{
  char link[FILENAME_MAX+1];
  unsigned char buf[65536];
  checksum_state c;
  digest_state s;
  struct stat sb;
  ssize_t n;
  int fd;

  checksum_init(&c);
  digest_init(&s);

  /* Check if path is a symlink or a normal file.*/
  if (lstat(path, &sb) == -1) return false;

  if (S_ISLNK(sb.st_mode)) {
    /* Checksum the contents of the symlink rather than what it points to. */
    if ((n = readlink(path, link, sizeof(link) - 1)) == -1) return false;
    checksum_update(c, (unsigned char*)link, n);
    digest_update(s, (unsigned char*)link, n);
  } else {
    /* Checksum the contents of the file. */
    if ((fd = open(path, O_RDONLY)) == -1) return false;
    while ((n = read(fd, buf, sizeof(buf))) > 0) {
      checksum_update(c, buf, n);
      digest_update(s, buf, n);
    }
    (void) close(fd);
  }

  checksum_final(c, checksum);
  digest_final(s, digest);

  return n != -1 && !signaled;
}
