/*
  update() is called by serve1() after an "update" command. It
  computes the signature (checksums) of a file and sends it to the
  client, then waits for the client to send it a delta (list of
  patches), which it uses to update the file. After a successful
  update, it logs the new modification time and size of the file.

  It can also take a "shortcut": If there is a file in the log with
  the same checksum, digest and size as the updated file will have, it
  can copy that file locally and doesn't need to send a signature and
  apply a delta.

  Implementation in protocol 1 and 2:

  The file is read as a series of blocks of blocksize bytes each (the
  last one may be shorter). For each block two checksums are computed:
  a 4-bytes fast but weak checksum and a 16-byte MD5 hash.

  For each block, a line is printed on stream "out" with the checksum,
  the hash, and the size of the block, the first two in hexadecimal,
  and separated by single spaces. The last line is a line with a
  single dot (".").

  Next, update() waits for and reads a delta from stream "in",
  consisting of literal bytes (in hexadecimal) and references to
  blocks in the existing file. The delta ends with a line consisting
  of a single dot. It creates a temporary file into which it copies
  the bytes and the referenced blocks from the existing file. When
  done, it overwrites (via rename()) the old file with the temporary
  file.
*/

#include "stdincls.h"
#include "getline.e"
#include "print.e"
#include "types.e"
#include "errcodes.e"
#include "s-sumfile.e"
#include "s-store.e"
#include "s-checksum.e"
#include "s-digest.e"


/* Convert a base64 character to bits, or 64 if error */
static unsigned char base64[] = {
  64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, /* 00-0F */
  64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, /* 10-1F */
  64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 62, 64, 64, 64, 63, /* space-/ */
  52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 64, 64, 64, 64, 64, 64, /* 0-? */
  64,  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, /* @-O */
  15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 64, 64, 64, 64, 64, /* P-_ */
  64, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, /* `-o */
  41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 64, 64, 64, 64, 64, /* p-7F */
  64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, /* 80-8F */
  64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, /* 90-9F */
  64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, /* A0-AF */
  64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, /* B0-BF */
  64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, /* C0-CF */
  64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, /* D0-DF */
  64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, /* E0-EF */
  64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, /* F0-FF */
};


/* generate_signature -- compute checksums, write them on out */
static bool generate_signature(FILE *out, FILE *oldfile, size_t blocksize)
{
  unsigned char m[DIGEST_LEN], buf[blocksize];
  uint32_t sum;
  size_t n;

  if (oldfile)
    /* Loop over the blocks of oldfile, computing checksums */
    while ((n = fread(buf, 1, blocksize, oldfile)) > 0) {
      sum = checksum(buf, blocksize, 0, n);
      digest(buf, blocksize, 0, n, m);
      print(out, "%x %s %lu\n", sum, digest_to_str(m), n);
    }

  debug("finished sending signature: %s\n",
	!oldfile || feof(oldfile) ? "OK" : strerror(errno));

  /* End with "." or an error. A non-existent file is treated as empty. */
  if (!oldfile || feof(oldfile)) {
    print1(out, ".\n");
    return true;
  } else {
    print(out, "? %03d %s\n", EC_SERVER + errno, strerror(errno));
    return false;
  }
}


/* read_and_apply_delta -- read the delta and update oldfile with it */
static bool read_and_apply_delta(FILE *in, FILE *out, FILE *oldfile,
				 FILE *tmpfile, size_t blocksize)
{
  unsigned char buf[blocksize], *bytes = NULL;
  long blocknumber, count;
  char *line, c = 0, h;
  size_t n, j;
  bool ok = true;
  int i;

  /* Read the delta line by line and construct tmpfile with it */
  while ((line = getline1(in)) && line[0] != '.' && line[0] != '?') {

    if (line[0] != '*') {	/* A line of literal bytes */

      /* Decode base64 and write to tmpfile */
      if ((bytes = realloc(bytes, strlen(line)))) /* More than enough space */
	for (i = 0, j = 0; (h = base64[(unsigned char)line[i]]) != 64; i++)
	  switch (i & 0x3) {
	  case 0: c = h << 2; break; /* Store 6 bits */
	  case 1: bytes[j++] = c | (h >> 4); c = (h & 0xF) << 4; break;
	  case 2: bytes[j++] = c | (h >> 2); c = (h & 0x3) << 6; break;
	  case 3: bytes[j++] = c | h; break;
	  }
      if (ok && bytes) ok = fwrite(bytes, 1, j, tmpfile) == j;

    } else if (!oldfile) {
      print(out, "? %03d Delta references non-existent file\n", EC_INVDELTA);
      free(bytes);
      return false;

    } else {			/* A line of block references */

      /* Parse numbers+counts, copy blocks from oldfile to
	 tmpfile. The first block has number 1. */
      char *p = line;
      while (ok && (blocknumber = strtol(p + 1, &p, 10)) > 0) {
	count = *p == '+' ? strtol(p + 1, &p, 10) : 0;
	ok = fseek(oldfile, (blocknumber - 1) * blocksize, SEEK_SET) == 0;
	for (i = 0; ok && i <= count; i++)
	  ok = (n = fread(buf, 1, blocksize, oldfile)) != 0 &&
	    fwrite(buf, 1, n, tmpfile) == n;
      }
    }
  }
  free(bytes);

  debug("finished reading the delta: %s\n",
	!ok || !line ? strerror(errno) :
	line && line[0] == '.' ? "OK" :
	line + 2);

  /* Print error message if there was an error on our side. */
  if (!ok) print(out, "? %03d %s\n", EC_SERVER + errno, strerror(errno));

  return ok && line && line[0] == '.';
}


/* shortcut -- copy an existing local file to tmpfile */
static bool shortcut(FILE *tmpfile, const char *root, fileinfo info)
{
  char path[FILENAME_MAX+1], buf[65536];
  struct stat statbuf;
  FILE *f = NULL;
  ssize_t n;

  /* Open the existing file, lock it, see if it hasn't changed. */
  if (!make_absolute(root, info.path, path) ||
      !(f = fopen(path, "rb")) ||
      flock(fileno(f), LOCK_EX) == -1 ||
      fstat(fileno(f), &statbuf) == -1 ||
      statbuf.st_mtime != info.time ||
      statbuf.st_size != info.size) {
    if (f) fclose(f);
    return false;
  }

  /* Copy the existing file to tmpfile */
  while ((n = fread(buf, 1, sizeof(buf), f)) > 0 &&
	 fwrite(buf, 1, n, tmpfile) > 0);

  /* Return true if copy was successful */
  return !ferror(f) && fclose(f) == 0 && !ferror(tmpfile);
}


/* update -- send a signature, read a delta, update the file and log it */
EXPORT void update(FILE *in, FILE *out, const char *root, const DB store,
		   const size_t blocksize, fileinfo info,
		   const bool case_sensitive)
{
  char path[FILENAME_MAX+1], tmpname[FILENAME_MAX+16];
  struct itimerval timer = {{0, 0}, {10, 0}};
  struct itimerval timer0 = {{0, 0}, {0, 0}};
  FILE *oldfile = NULL, *tmpfile = NULL;
  bool have_shortcut, used_shortcut;
  struct timeval times[2];
  fileinfo shortcutinfo;
  int r;

  if (!make_absolute(root, info.path, path)) {
    print(out, "? %03d File name too long\n", EC_TOOLONG);
    return;
  }

  /* Try to open the old file. It's OK if it doesn't exist */
  if (!(oldfile = fopen(path, "rb")) && errno != ENOENT) {
    print(out, "? %03d %s\n", EC_SERVER + errno, strerror(errno));
    return;
  }

  /* Lock the file, if any, until we're done writing the new file. */
  if (oldfile) {
    setitimer(ITIMER_REAL, &timer, NULL); /* Wait no more than 10 s */
    r = flock(fileno(oldfile), LOCK_EX);
    setitimer(ITIMER_REAL, &timer0, NULL);
    if (r == -1) {
      print(out, "? %03d %s\n", EC_SERVER + errno, strerror(errno));
      (void) fclose(oldfile);
      return;
    }
  }

  /* See if we have an identical local file. */
  have_shortcut = info.status == '+' &&
    store_get_by_sum(store, &info.sums, &shortcutinfo) &&
    shortcutinfo.size == info.size;

  debug("Shortcut for %s: %s\n", info.path,
	have_shortcut ? shortcutinfo.path : "(none)");

  /* If the "identical file" is the file itself, it means its
     contents haven't changed, but its modification time has. */
  if (have_shortcut &&
      (case_sensitive ?
       strcmp(info.path, shortcutinfo.path) == 0 :
       strcasecmp(info.path, shortcutinfo.path) == 0) &&
      oldfile) {

    times[0].tv_sec = time(NULL); times[0].tv_usec = 0;
    times[1].tv_sec = info.time; times[1].tv_usec = 0;
    assert(info.mode <= 07777);
    if (futimes(fileno(oldfile), times) == -1 ||
	fchmod(fileno(oldfile), info.mode) == -1 ||
	(oldfile && fclose(oldfile) == -1)) {
      print(out, "? %03d %s\n", EC_SERVER + errno, strerror(errno));
      return;
    }

    /* Log the new mode, modification time and size. Return success. */
    store_del(store, info.path);
    info.mode |= S_IFREG;	/* Add the file type */
    (void) store_put(store, info); /* What to do of it fails? */
    print(out, "? %03d Shortcut: file time updated\n", EC_SHORTCUT);
    return;
  }

  /* If file itself doesn't exist, make sure its directory exists.
     Create a file ".r2sync-XXXXXX" in that directory. */
  if ((!oldfile && !create_directories(path)) ||
      !(tmpfile = create_temp_file(path, tmpname))) {
    print(out, "? %03d %s\n", EC_SERVER + errno, strerror(errno));
    if (oldfile) (void) fclose(oldfile);
    return;
  }

  /* If we have an identical local file, try to copy it to the tmpfile. */
  used_shortcut = have_shortcut && shortcut(tmpfile, root, shortcutinfo);

  /* Otherwise, do the rsync algorithm: Output a checksum, strong hash
     and size for each blocksize bytes. Then read the delta and
     construct tmpfile with it. */
  if (!used_shortcut)
    if (!generate_signature(out, oldfile, blocksize) ||
	!read_and_apply_delta(in, out, oldfile, tmpfile, blocksize)) {
      if (oldfile) (void) fclose(oldfile);
      (void) remove(tmpname);
      (void) fclose(tmpfile);
      return;
    }

  /* Rename tmpfile to path, set modification time and mode bits,
     close the files (releasing the lock). Delete (unlink) any
     existing file first, so that on a case-insensitive,
     case-preserving file system it gets the name as it is spelled on
     the other target.

     TODO: Get the file size with "long filesize = ftell(tmpfile)" or
     trust that info.size is correct?
  */
  times[0].tv_sec = time(NULL); times[0].tv_usec = 0;
  times[1].tv_sec = info.time; times[1].tv_usec = 0;
  assert(info.mode <= 07777);
  if (fflush(tmpfile) == -1 ||
      (unlink(path) == -1 && errno != ENOENT) ||
      rename(tmpname, path) == -1 ||
      futimes(fileno(tmpfile), times) == -1 ||
      fchmod(fileno(tmpfile), info.mode) == -1 ||
      (oldfile && fclose(oldfile) == -1) ||
      fclose(tmpfile) == -1) {
    print(out, "? %03d %s\n", EC_SERVER + errno, strerror(errno));
    return;
  }

  /* Success. Log the new mode, modification time and size. Return "OK". */
  store_del(store, info.path);
  info.mode |= S_IFREG;		 /* Add the file type */
  (void) store_put(store, info); /* What to do of it fails? */

  if (!used_shortcut) print1(out, "OK\n");
  else print(out, "? %03d Shortcut: copied from %s\n", EC_SHORTCUT,
	     shortcutinfo.path);
}
